<?php
/**
 * WarForms Reboot - Entries DB Layer
 *
 * Step 2: Centralize DB install/upgrade + CRUD helpers in one file.
 *
 * NOTE: Table creation/migrations via dbDelta arrive in Step 4.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get current schema version stored in the DB.
 *
 * @return int
 */
function warforms_reboot_get_schema_version() {
	return (int) get_option( 'warforms_reboot_schema_version', 0 );
}

/**
 * Set schema version stored in the DB.
 *
 * @param int $version
 * @return bool
 */
function warforms_reboot_set_schema_version( $version ) {
	return (bool) update_option( 'warforms_reboot_schema_version', (int) $version );
}

/**
 * Install / upgrade routine.
 *
 * Step 3 wires this to plugin activation.
 * Step 4 runs dbDelta() and creates/migrates the entries table.
 *
 * @return void
 */
function warforms_reboot_install() {
	$desired = defined( 'WARFORMS_REBOOT_DB_VERSION' ) ? (int) WARFORMS_REBOOT_DB_VERSION : 1;
	$current = warforms_reboot_get_schema_version();

	// Only run dbDelta when we need to install/upgrade.
	if ( $current < $desired ) {
		global $wpdb;

		// dbDelta is defined here.
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$table_name      = function_exists( 'warforms_reboot_entries_table' )
			? warforms_reboot_entries_table()
			: $wpdb->prefix . 'warforms_entries';
		$charset_collate = $wpdb->get_charset_collate();

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$sql = "CREATE TABLE {$table_name} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			form_id bigint(20) unsigned NOT NULL,
			created_at datetime NOT NULL,
			user_id bigint(20) unsigned DEFAULT NULL,
			ip varchar(45) DEFAULT NULL,
			payload_json longtext NOT NULL,
			PRIMARY KEY  (id),
			KEY form_id (form_id),
			KEY created_at (created_at)
		) {$charset_collate};";

		dbDelta( $sql );

		// Record that the install/upgrade has been applied.
		warforms_reboot_set_schema_version( $desired );
	}
}

// --- CRUD helpers will live below (Step 6+ will start adding them) ---
/**
 * Insert a form entry into the entries table.
 *
 * Step 6: warforms_reboot_insert_entry()
 *
 * @param int   $form_id
 * @param array $values
 * @param array $raw_context
 * @return int|\WP_Error Entry ID on success, WP_Error on failure.
 */
function warforms_reboot_insert_entry( $form_id, array $values, array $raw_context = [] ) {
	if ( ! function_exists( 'warforms_reboot_entries_table' ) ) {
		return new \WP_Error( 'missing_table_helper', 'Entries table helper is missing.' );
	}

	$form_id = (int) $form_id;
	if ( $form_id <= 0 ) {
		return new \WP_Error( 'invalid_form_id', 'Invalid form id.' );
	}

	// Build payload: { values, context }
	$payload = [
		'values'  => $values,
		'context' => $raw_context,
	];

	// Step 7: addon extension points around payload
	$payload = apply_filters( 'warforms_entry_payload', $payload, $form_id, $values );
	do_action( 'warforms_before_entry_insert', $payload, $form_id, $values );

	$payload_json = wp_json_encode( $payload, JSON_UNESCAPED_SLASHES );
	if ( ! is_string( $payload_json ) || $payload_json === '' ) {
		return new \WP_Error( 'json_encode_failed', 'Failed to encode entry payload.' );
	}

	$user_id = 0;
	if ( isset( $raw_context['user_id'] ) ) {
		$uid = (int) $raw_context['user_id'];
		if ( $uid > 0 ) {
			$user_id = $uid;
		}
	}
$ip = null;
	if ( isset( $raw_context['ip'] ) ) {
		$ip_candidate = sanitize_text_field( (string) $raw_context['ip'] );
		$ip_candidate = substr( $ip_candidate, 0, 45 );
		if ( $ip_candidate !== '' ) {
			$ip = $ip_candidate;
		}
	}

	global $wpdb;

	$data = [
		'form_id'      => $form_id,
		'created_at'   => current_time( 'mysql', true ), // GMT
		'user_id'      => $user_id,
		'ip'           => $ip,
		'payload_json' => $payload_json,
	];

	$formats = [ '%d', '%s', '%d', '%s', '%s' ];

	$ok = $wpdb->insert( warforms_reboot_entries_table(), $data, $formats );
	if ( false === $ok ) {
		// Expose db detail to callers via WP_Error (and store in diagnostics in the submit handler).
		$detail = is_string( $wpdb->last_error ) ? $wpdb->last_error : '';
		return new \WP_Error( 'db_insert_failed', 'Failed to save entry.', [ 'wpdb_last_error' => $detail ] );
	}

	return (int) $wpdb->insert_id;
}



/**
 * Step 13: Query entries (paged) for admin listing.
 *
 * @param int $form_id
 * @param int $page
 * @param int $per_page
 * @return array{rows: array<int,object>, total: int}
 */
function warforms_reboot_get_entries( $form_id = 0, $page = 1, $per_page = 20 ) {
	global $wpdb;

	$form_id  = (int) $form_id;
	$page     = max( 1, (int) $page );
	$per_page = (int) $per_page;
	if ( $per_page < 1 ) {
		$per_page = 20;
	}
	// Hard cap to avoid heavy accidental queries.
	if ( $per_page > 200 ) {
		$per_page = 200;
	}

	$offset = ( $page - 1 ) * $per_page;
	$table  = function_exists( 'warforms_reboot_entries_table' )
		? warforms_reboot_entries_table()
		: $wpdb->prefix . 'warforms_entries';

	$where = '1=1';
	$args  = [];

	if ( $form_id > 0 ) {
		$where .= ' AND form_id = %d';
		$args[] = $form_id;
	}

	// Total count.
	$sql_count = "SELECT COUNT(*) FROM {$table} WHERE {$where}";
	if ( ! empty( $args ) ) {
		$total = (int) $wpdb->get_var( $wpdb->prepare( $sql_count, $args ) );
	} else {
		$total = (int) $wpdb->get_var( $sql_count );
	}

	if ( ! empty( $wpdb->last_error ) && function_exists( 'warforms_reboot_set_last_error' ) ) {
		warforms_reboot_set_last_error( 'db', (string) $wpdb->last_error );
	}

	// Rows.
	$sql_rows = "SELECT id, form_id, created_at, user_id, ip
		FROM {$table}
		WHERE {$where}
		ORDER BY created_at DESC, id DESC
		LIMIT %d OFFSET %d";

	$args_rows = array_merge( $args, [ $per_page, $offset ] );
	$rows      = $wpdb->get_results( $wpdb->prepare( $sql_rows, $args_rows ) );
	if ( ! empty( $wpdb->last_error ) && function_exists( 'warforms_reboot_set_last_error' ) ) {
		warforms_reboot_set_last_error( 'db', (string) $wpdb->last_error );
	}
	if ( ! is_array( $rows ) ) {
		$rows = [];
	}

	return [
		'rows'  => $rows,
		'total' => $total,
	];
}

/**
 * Fetch a single entry row by ID.
 *
 * Step 15/16 support: used by the admin "View Entry" screen.
 *
 * @param int $entry_id
 * @return object|null WPDB row object on success, null if not found.
 */

/**
 * Get per-form entry stats for admin UX (counts + last submission date).
 *
 * @return array<int, array{total:int,last_created_at:string}>
 */
function warforms_reboot_get_entry_stats_by_form() {
	global $wpdb;

	$table = function_exists( 'warforms_reboot_entries_table' )
		? warforms_reboot_entries_table()
		: $wpdb->prefix . 'warforms_entries';

	// form_id => [total => int, last_created_at => 'YYYY-mm-dd HH:ii:ss' (GMT)]
	$stats = [];

	$rows = $wpdb->get_results( "SELECT form_id, COUNT(*) AS total, MAX(created_at) AS last_created_at FROM {$table} GROUP BY form_id" );
	if ( ! empty( $wpdb->last_error ) && function_exists( 'warforms_reboot_set_last_error' ) ) {
		warforms_reboot_set_last_error( 'db', (string) $wpdb->last_error );
	}

	if ( is_array( $rows ) ) {
		foreach ( $rows as $r ) {
			$fid = isset( $r->form_id ) ? (int) $r->form_id : 0;
			if ( $fid <= 0 ) {
				continue;
			}
			$stats[ $fid ] = [
				'total'          => isset( $r->total ) ? (int) $r->total : 0,
				'last_created_at' => isset( $r->last_created_at ) ? (string) $r->last_created_at : '',
			];
		}
	}

	return $stats;
}


function warforms_reboot_get_entry( $entry_id ) {
	global $wpdb;

	$entry_id = (int) $entry_id;
	if ( $entry_id <= 0 ) {
		return null;
	}

	$table = function_exists( 'warforms_reboot_entries_table' )
		? warforms_reboot_entries_table()
		: $wpdb->prefix . 'warforms_entries';

	$row = $wpdb->get_row(
		$wpdb->prepare(
			"SELECT id, form_id, created_at, user_id, ip, payload_json FROM {$table} WHERE id = %d",
			$entry_id
		)
	);

	if ( ! empty( $wpdb->last_error ) && function_exists( 'warforms_reboot_set_last_error' ) ) {
		warforms_reboot_set_last_error( 'db', (string) $wpdb->last_error );
	}

	return is_object( $row ) ? $row : null;
}

